{ -----------------------------------------------------------------------------
   sDesk 0.xxx - Semik's desktop, virtual desktops for MS-Windows
   Copyright (C) 1998  Jan Tomasek <xtomasej@fel.cvut.cz>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

History:
   31.10.1998 Released under GNU
              Added code for NC like screen saver activation
------------------------------------------------------------------------------ }
{$LONGSTRINGS OFF}
unit Main;

interface

uses
  Windows,  Messages, SysUtils, Classes,
  Graphics, Controls, Forms,    Dialogs,
  ExtCtrls, Menus,    ComCtrls, RXShell,
  DebugW,   Registry;

Const
  MSG_TABKEY = WM_USER + 9996;

type
  { 15.07.1999 added by Tim Boughen }
  THiddenListItem = class(Tobject)
  private
    FWndHandle : THandle;
    FActive    : boolean;
    // JAN FTitle     : ShortString;
    // JAN FRct       : TRect;
    function RTitle:String;
  public
    property WndHandle: THandle read FWndHandle write FWndHandle;
    property Active: boolean read FActive write FActive;
    property Title: ShortString read RTitle;
    // JAN property Rct: TRect read FRct write FRct;
  end;
  { end Tim }

  TfMain = class(TForm)
    UpdateTimer     : TTimer;
    HideTimer       : TTimer;
    KeyAck          : TTimer;
    pmContextMenu   : TPopupMenu;
    miMoveHere      : TMenuItem;
    miOptions       : TMenuItem;
    miAbout         : TMenuItem;
    TrayIcon        : TRxTrayIcon;
    ScreenSaverTimer: TTimer;
    AllowMove: TTimer;
    miSwitchTo: TMenuItem;
    miN1: TMenuItem;
    miN2: TMenuItem;
    AutoSwitch: TTimer;
    miN3: TMenuItem;
    miExit: TMenuItem;
    miStickyWindows: TMenuItem;
    MouseSwitch: TTimer;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure FormPaint(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure FormMouseUp(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure FormMouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure FormMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
    procedure UpdateTimerTimer(Sender: TObject);
    procedure HideTimerTimer(Sender: TObject);
    procedure KeyAckTimer(Sender: TObject);
    procedure miAboutClick(Sender: TObject);
    procedure miOptionsClick(Sender: TObject);
    procedure TrayIconClick(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure ScreenSaverTimerTimer(Sender: TObject);
    procedure AllowMoveTimer(Sender: TObject);
    procedure AutoSwitchTimer(Sender: TObject);
    procedure miExitClick(Sender: TObject);
    procedure StickyWindowsClick(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure MouseSwitchTimer(Sender: TObject);
  private
    { Private declarations }
    FDesk               : Word;          // actual desktop no
    WindowsList         : TList;         // fillin by GetVisibleWindows
    FHiddenWindowsList  : TList;
    ActiveWindow        : THandle;
    MovingWindow        : Boolean;
    MWindow             : THandle;
    MWStartWPos         : TPoint;
    MWStartMPos         : TPoint;
    Desktop             : TBitmap;       // for double buffering
    popX, popY          : Integer;
    IgnoreSetFocus      : Boolean;
    IsScreenSaverActive : Boolean;
    ScreenSaverTickCount: Integer;       // must be = 2 for screen saver activation
    ScreenSaverLastPos  : TPoint;
    LastWindowHandle    : THandle;
    AllowDeskSwitch     : Boolean;
    OptionsProc         : Boolean;
    FHiddenReg          : TRegistry;
    FPopMenuFromTrayIcon: Boolean;
    { Used for storing handle of active window, used in WDesk }
    FRemeberActiveWnd   : TList;
    { Used for switching by mouse fature }
    FMouseSwitchCounter : Integer;
    FLastCusorPos       : TPoint;

    procedure GetVisibleWindows;
    procedure OnlyFromHere;
    procedure MoveWindows(X,Y:Integer);
    { 16.07.99 added by Tim Boughen }
    procedure ResetWindowStyles(TargetDestop:Integer);
    procedure RestoreHiddenApps;
    procedure DeleteHiddenRegistry(W:THandle);
    procedure RestoreHiddenRegistry;
    procedure CreateHiddenRegistry(W:THandle);
    { end Tim }
    procedure PaintWindows;
    procedure WDesk(aDesk:Word);
    procedure WMNCMOUSEMOVE(Var Msg:TMsg); message WM_NCMOUSEMOVE;
    procedure WMHOTKEY(Var Msg:TMsg); message WM_HOTKEY;
    procedure WndProc(var Message: TMessage); override;

    procedure RealocateRemeberActive;
    Procedure MoveHereClick(Sender: TObject);
    Procedure SwitchToClick(Sender: TObject);
    procedure RegKeys;
    procedure URegKeys;
    procedure SetParams;
    //procedure AddToLog(Const S:String);
    function  CountDesktopNo(W:THandle):Integer;
    //function  IsValidWindow(W:THandle):Boolean;
    function DockWindow:Boolean;
    procedure UnDockWindow;
    procedure MoveAllToFirst;
    Procedure MyMoveWindow(W:THandle;aDesk:Word);

  public
    { Public declarations }
    function FindHandleInHiddenList(H: THandle): integer;
    property Desk:Word read FDesk write WDesk;
  end;

var
  fMain: TfMain;
  fHidden: THiddenListItem;

Const
  hkContextMenu    = 200;
  hkDeskPlus       = 201;
  hkDeskMinus      = 202;
  hkDeskPlusRow    = 203;
  hkDeskMinusRow   = 204;
  hkDesk           = 205;

implementation
{$R *.DFM}
Uses
  Options, About, Win2Tree,
  Config,  ShellAPI, Sticky, StickyProp, DeskName;

Const
  DeskFontSize     = 5;
  DeskFontName     = 'Small Fonts';
  clDeskWindow     = clBtnHighlight;
  clDeskAWindow    = clWindow;
  clDeskWindowFrame= clBtnText;
  RegKey           = 'Software\Jan Tomasek\sDesk';
  { 16.07.99 added by Tim Boughen }
  RegKey_HiddenWnd = RegKey+'\HiddenApps';
  { End Tim }

  UM_ENABFOCUS     = WM_USER+10001;
{-$I sDeskHook.inc}

Function IsInRect(Rect:TRect; X, Y:Integer):Boolean; 
Begin
  Result:=(Rect.Left<=X)and(Rect.Right>X)and(Rect.Top<=Y)and(Rect.Bottom>Y);
End;

// -----------------------------------------------------------------------------
// This code was taken from UDDF - Unofficial Delphi Developers FAQ,
// url http://www.gnomehome.demon.nl/uddf/
function WinExecAndWait32(Const FileName:String; Visibility : integer):integer; 
var
  zAppName:array[0..512] of char;
  zCurDir:array[0..255] of char;
  WorkDir:String;
  StartupInfo:TStartupInfo;
  ProcessInfo:TProcessInformation;
begin
  StrPCopy(zAppName,FileName);
  GetDir(0,WorkDir);
  StrPCopy(zCurDir,WorkDir);
  FillChar(StartupInfo,Sizeof(StartupInfo),#0);
  StartupInfo.cb := Sizeof(StartupInfo);

  StartupInfo.dwFlags := STARTF_USESHOWWINDOW;
  StartupInfo.wShowWindow := Visibility;
  if not CreateProcess(nil,
    zAppName,                      { pointer to command line string }
    nil,                           { pointer to process security attributes }
    nil,                           { pointer to thread security attributes }
    false,                         { handle inheritance flag }
    CREATE_NEW_CONSOLE or          { creation flags }
    NORMAL_PRIORITY_CLASS,
    nil,                           { pointer to new environment block }
    nil,                           { pointer to current directory name }
    StartupInfo,                   { pointer to STARTUPINFO }
    ProcessInfo) then Result := -1 { pointer to PROCESS_INF }

  else begin
    WaitforSingleObject(ProcessInfo.hProcess,INFINITE);
    GetExitCodeProcess(ProcessInfo.hProcess,Result);
  end;
end;


// -----------------------------------------------------------------------------
procedure TfMain.FormCreate(Sender: TObject);
begin
  IgnoreSetFocus:=False;

  { 11.07.1999 added by Tim Boughen }
  If CFG.Window.ShowTitleBar Then BorderStyle := bsSizeToolWin
  Else BorderStyle := bsNone;
  { end Tim }
  Left                := CFG.Window.Left;
  Top                 := CFG.Window.Top;
  ClientWidth         := CFG.Window.xStep*CFG.Window.Columns;
  ClientHeight        := CFG.Window.yStep*CFG.Window.Rows;

  SetParams;

  // Create options window (for options)
  fOptions            := TfOptions.Create(self);

  TrayIcon.Active     := CFG.General.PutIconToTray;
  FDesk               := 0;
  MovingWindow        := False;

  WindowsList         := TList.Create;
  Desktop             := TBitmap.Create;
  IsScreenSaverActive := False;
  ScreenSaverTickCount:= 0;
  ScreenSaverLastPos  := Point(-5,-5);
  AllowDeskSwitch     := True;
  OptionsProc         := False;
  FHiddenWindowsList  := TList.Create;
  FHiddenReg          := TRegistry.Create;
  FRemeberActiveWnd   := TList.Create;
  FMouseSwitchCounter := 0;
  FLastCusorPos       := Point(-1, -1);

  { 19.07.99 Added by Tim Boughen }
  RestoreHiddenRegistry;
  { End Tim }

  { Hide apps from desktops<>0 }
  GetVisibleWindows;
  ResetWindowStyles(FDesk);

  if SwitchToDesk>=0 then 
    PostMessage(Handle, WM_HOTKEY, hkDesk+SwitchToDesk, 0);
end;// -- TfMain.FormCreate ----------------------------------------------------

// -----------------------------------------------------------------------------
procedure TfMain.FormDestroy(Sender: TObject);
begin
  {$IFDEF SDEBUG}try Assert(False) except on E:EAssertionFailed do DebugWAddEString(E.Message, 'TfMain.Destroyt: Begin', [0]); end;{$ENDIF}
  // Save cfg into registry
  CFG.Window.Left    := Left;
  CFG.Window.Top     := Top;

  WriteCFG(CFG);
  URegKeys;

  FHiddenWindowsList.Free; FHiddenWindowsList := nil;
  FHiddenReg.Free; FHiddenReg := nil;        
  FRemeberActiveWnd.Free; FRemeberActiveWnd := nil; 
  WindowsList.Clear; WindowsList.Free; WindowsList := nil;
  Desktop.Free; Desktop:= nil;
  {$IFDEF SDEBUG}try Assert(False) except on E:EAssertionFailed do DebugWAddEString(E.Message, 'TfMain.Destroyt: End', [0]); end;{$ENDIF}
end; //-- TfMain.FormDestroy ---------------------------------------------------

procedure TfMain.SetParams;
Begin
  RegKeys;
  Caption := CFG.Window.PagerTitle;
  ScreenSaverTimer.Interval := CFG.ScreenSaver.ActivationTime div 10;
  ScreenSaverTimer.Enabled := CFG.ScreenSaver.Allowed;
  MouseSwitch.Interval := CFG.DesktopSwitching.MouseSwSpeed div 3;
  MouseSwitch.Enabled  := CFG.DesktopSwitching.MouseSwEnabled;
End;

{procedure TfMain.AddToLog(Const S:String);
Begin
  If Hint<>'' Then
    Hint:=Hint+#13#10+S
  Else
    Hint:=S;
End;}

procedure RegHotKey(Handle:HWND;Const ShortCut:TShortCut; ID:Integer); 
Var
  Key  : Word;
  State: TShiftState;

  Modif: Word;
Begin
  ShortCutToKey(ShortCut, Key, State);
  Modif:=0;
  If ssShift in State Then Modif:=Modif or MOD_SHIFT;
  If ssAlt   in State Then Modif:=Modif or MOD_ALT;
  If ssCtrl  in State Then Modif:=Modif or MOD_CONTROL;
  RegisterHotkey(Handle, ID, Modif, Key);
End;

procedure TfMain.RegKeys;
Var
  I : Integer;
Begin
  RegHotKey(Handle, CFG.General.ContextMenu, hkContextMenu);
  RegHotKey(Handle, CFG.HotKeys.DeskPlus, hkDeskPlus);
  RegHotKey(Handle, CFG.HotKeys.DeskMinus, hkDeskMinus);
  RegHotKey(Handle, CFG.HotKeys.DeskPlusRow, hkDeskPlusRow);
  RegHotKey(Handle, CFG.HotKeys.DeskMinusRow, hkDeskMinusRow);
  For I:=0 To CFG.Window.Columns*CFG.Window.Rows-1 Do
    RegHotKey(Handle, CFG.HotKeys.DeskDirect[I], hkDesk+I);
End;{-- SetParams -------------------------------------------------------------}

procedure TfMain.URegKeys;
Var
  I : Integer;
Begin
  UnregisterHotKey(Handle, hkContextMenu);
  UnregisterHotKey(Handle, hkDeskPlus);
  UnregisterHotKey(Handle, hkDeskMinus);
  UnregisterHotKey(Handle, hkDeskPlusRow);
  UnregisterHotKey(Handle, hkDeskMinusRow);
  For I:=0 To CFG.Window.Columns*CFG.Window.Rows-1 Do
    UnregisterHotKey(Handle, hkDesk+I);
End;

procedure TfMain.FormPaint(Sender: TObject);
Var
  I, J  : Word;
begin
  If Desktop<>nil Then With Desktop.Canvas Do Begin
    Desktop.Width:= ClientWidth+1;
    Desktop.Height:=ClientHeight+1;

    // clear space;
    Brush.Style    := bsSolid;
    Brush.Color    := CFG.Window.InactiveColor;

    FillRect(Rect(0,0,ClientWidth,ClientHeight));

    // paint active desktop
    Brush.Color    := CFG.Window.ActiveColor;
    FillRect(Rect(Round(CFG.Window.xStep*(FDesk mod CFG.Window.Columns)),  Round(CFG.Window.yStep*(FDesk div CFG.Window.Columns)),
                  Round(CFG.Window.xStep*(FDesk mod CFG.Window.Columns+1)),Round(CFG.Window.yStep*(FDesk div CFG.Window.Columns+1))));

    // paint desktops separators
    Pen.Color      := CFG.Window.GridColor;
    For I:=0 To CFG.Window.Columns Do Begin
      MoveTo(Round(I*CFG.Window.xStep)-1*Byte(I=CFG.Window.Columns),0);
      LineTo(Round(I*CFG.Window.xStep)-1*Byte(I=CFG.Window.Columns),ClientHeight);
    End;
    For I:=0 To CFG.Window.Rows Do Begin
      MoveTo(0,Round(I*CFG.Window.yStep)-1*Byte(I=CFG.Window.Rows));
      LineTo(ClientWidth,Round(I*CFG.Window.yStep)-1*Byte(I=CFG.Window.Rows));
    End;

    // write desktop numbers
    if CFG.Window.ShowDesktopNumbers then begin
      Font.Name  := DeskFontName;
      Font.Size  := DeskFontSize;
      Brush.Style:= bsDiagCross;           // Invisible background ??
      For I:=0 To CFG.Window.Columns-1 Do
        For J:=0 To CFG.Window.Rows-1 Do Begin
          TextOut(2+Round(I*CFG.Window.xStep), 1+Round(J*CFG.Window.yStep), IntToStr(1+I+J*CFG.Window.Columns)+'.');
        End;
    End;

    // paint windows
    PaintWindows;
  End;
  BitBlt(Canvas.Handle, 0, 0, ClientWidth, ClientHeight, Desktop.Canvas.Handle, 0, 0, SRCCOPY);
end;

procedure TfMain.PaintWindows;
var
  I     : Integer;
  WRct  : TRect;
  W     : THandle;
  xScale,
  yScale: Real;
  S     : ShortString;
  FWnd  : THandle;
begin
  GetVisibleWindows;
  FWnd := GetForegroundWindow;
  If Desktop<>nil then With Desktop.Canvas Do Begin
    xScale := CFG.Window.xStep / Screen.Width;
    yScale := CFG.Window.yStep / Screen.Height;

    Font.Name  :=DeskFontName;
    Font.Size  :=DeskFontSize;

    If WindowsList.Count>0 Then For I:=WindowsList.Count-1 DownTo 0 Do Begin
      W := THandle(WindowsList[I]);
      S := GetWindowTitle(W);
      GetWindowRect(W,WRct);
      Inc(WRct.Left,  (FDesk mod CFG.Window.Columns)*Screen.Width);
      Inc(WRct.Right, (FDesk mod CFG.Window.Columns)*Screen.Width);

      Inc(WRct.Top,   (FDesk div CFG.Window.Columns)*Screen.Height);
      Inc(WRct.Bottom,(FDesk div CFG.Window.Columns)*Screen.Height);

      S:=GetWindowTitle(W);
      WRct := Rect(Round(WRct.Left*xScale),  Round(WRct.Top*yScale),
                   Round(WRct.Right*xScale), Round(WRct.Bottom*yScale));

      Brush.Style:=bsSolid;
      If W=FWnd Then Brush.Color:=clDeskAWindow
      Else Brush.Color:=clDeskWindow;
      Pen.Color  :=clDeskWindowFrame;
      FillRect(WRct);

      Rectangle(WRct.Left, WRct.Top, WRct.Right, WRct.Bottom);
      Brush.Style:=bsClear;
      TextRect(WRct, WRct.Left+2, WRct.Top, S);
    End;
  End;
end;

{Added By Tim Boughen }
procedure TfMain.RestoreHiddenApps;
var
  i       : integer;
  newItem : THiddenListItem;
Begin
  for i := 0 to FHiddenWindowsList.Count - 1 do begin
      newItem := FHiddenWindowsList.Items[i];
      ShowWindow(newItem.WndHandle, SW_RESTORE);
      EnableWindow(newItem.WndHandle, True);
      DeleteHiddenRegistry(newItem.WndHandle);
      THiddenListItem(FHiddenWindowsList.Items[i]).Destroy;
  end;
end;

procedure TfMain.ResetWindowStyles(TargetDestop:Integer); 
var
  I, Pos        : Integer;
  Hide          : Boolean;
  DeskNameWnd   : Boolean;
  W             : THandle;
  newItem       : THiddenListItem;
  VWP           : TWindowPlacement;
begin
   If WindowsList.Count>0 Then Begin
     For I:=WindowsList.Count-1 DownTo 0 Do Begin
       W := THandle(WindowsList[I]);
       GetWindowPlacement(W, @VWP);

       if Assigned(fDesktopName)and(not Application.Terminated) then DeskNameWnd:=fDesktopName.Handle=W else DeskNameWnd:=False;
       Hide := (VWP.showCmd <> SW_MINIMIZE) and
               (VWP.showCmd <> SW_SHOWMINIMIZED) and
               (VWP.showCmd <> SW_SHOWMINNOACTIVE) and
               (VWP.showCmd <> SW_SHOWMAXIMIZED) and
               (CountDesktopNo(W)<>TargetDestop) and
               (not DeskNameWnd) and
               (not CFG.StickyWindows.List.IsSticky(W));

       Pos := FindHandleInHiddenList(W);
       If Hide Then Begin If Pos = -1 Then Begin
         newItem := THiddenListItem.Create;
         newItem.WndHandle := W;
         if W = ActiveWindow then newItem.Active:=True Else newItem.Active:=False;
         FHiddenWindowsList.Add(newItem);
         CreateHiddenRegistry(W);
         EnableWindow(W, False);
         ShowWindow(W, SW_HIDE);
       End End Else If Pos > -1 Then Begin
         DeleteHiddenRegistry(W);
         newItem := FHiddenWindowsList.Items[Pos];
         ShowWindow(W, SW_RESTORE);
         EnableWindow(W, True);
         If CFG.DesktopSwitching.ResetApps and newItem.Active Then SetForegroundWindow(W);
         newItem.Destroy;
         FHiddenWindowsList.Items[Pos]:=nil;
       End;
     End;
     FHiddenWindowsList.Pack;
   End;
end;

Procedure TfMain.RestoreHiddenRegistry;
Var
//   Reg        : TRegistry;
   HiddenKeys : TStrings;
   I,
   Code       : integer;
//   StartKey,
   HiddenKey  : string;
   HKT        : ShortString;
   HKH        : string;
   WH         : THandle;
   WT         : ShortString;
   Recovered  : integer;
   RecoverMsg : string;
begin
   Recovered := 0;
//   Reg := TRegistry.Create;
   HiddenKeys := TStringList.Create;
//   StartKey := Reg.CurrentPath;
   HiddenKey := RegKey_HiddenWnd;
   FHiddenReg.OpenKey(HiddenKey, False);

   FHiddenReg.GetValueNames(HiddenKeys);
   for I := 0 to HiddenKeys.Count -1 do begin
      HKH := HiddenKeys.Strings[I];
      HKT := FHiddenReg.ReadString(HKH);
      val(HKH, WH, Code);
      WT:=GetWindowTitle(WH);
      If WT = HKT then Begin
         Recovered:=Recovered+1;
         ShowWindow(WH, SW_RESTORE);
         EnableWindow(WH, True);
      end;
      FHiddenReg.DeleteValue(HKH);
   End;
   FHiddenReg.CloseKey;
//   Reg.Free;
   RecoverMsg:=Format('sDesk recovered %d hidden applications', [Recovered]);
   If Recovered > 0 then ShowMessage(RecoverMsg);
End;


Procedure TfMain.DeleteHiddenRegistry(W:THandle); 
Var
//   Reg        : TRegistry;
//   StartKey,
   HiddenKey,
   HiddenName : string;
begin
//   FHiddenReg := TRegistry.Create;
//   StartKey := FHiddenReg.CurrentPath;
   HiddenKey := RegKey_HiddenWnd;
   HiddenName := IntToStr(W);
   FHiddenReg.OpenKey(HiddenKey, True);
   FHiddenReg.DeleteValue(HiddenName);
   FHiddenReg.CloseKey;
//   Reg.Free;
end;

Procedure TfMain.CreateHiddenRegistry(W:THandle); 
Var
//   Reg        : TRegistry;
//   StartKey,
   HiddenKey,
   HiddenName : string;
begin
//   Reg := TRegistry.Create;
//   StartKey := Reg.CurrentPath;
   HiddenKey := RegKey_HiddenWnd;
   HiddenName := IntToStr(W);
   if not FHiddenReg.OpenKey(HiddenKey, True) then raise ERegistryException.CreateFmt('Cannot create key : %s', [HiddenKey]);
   FHiddenReg.WriteString(HiddenName, GetWindowTitle(W));
   FHiddenReg.CloseKey;
//   Reg.Free;
end;

function TfMain.FindHandleInHiddenList(H: THandle): integer; register; 
var
   i   : integer;
   HLI : THiddenListItem;
begin
  result := -1;
  if CFG.DesktopSwitching.HideApps then Begin
     for i := 0 to fMain.FHiddenWindowsList.Count - 1 do begin
         HLI := THiddenListItem(fMain.FHiddenWindowsList.Items[i]);
         if HLI<>nil Then if (HLI.WndHandle = H) then begin
            result := i;
            break;
         end;
     end;
  end;
end;

procedure TfMain.FormResize(Sender: TObject);                
begin
  If Visible and not OptionsProc Then Begin
    CFG.Window.xStep := Round(ClientWidth /CFG.Window.Columns);
    CFG.Window.yStep := Round(ClientHeight /CFG.Window.Rows);
    ClientWidth    := CFG.Window.Columns*CFG.Window.xStep;
    ClientHeight   := CFG.Window.Rows*CFG.Window.yStep;
    Invalidate;
  End;
end;

procedure TfMain.GetVisibleWindows;
Var
  W           : THandle;
  WndTitle    : ShortString;
  NotDeskName : Boolean;
Begin
  WindowsList.Clear;
  ActiveWindow:=GetForegroundWindow();
  W:=GetWindow(GetDesktopWindow,GW_CHILD);
  While W<>0 Do Begin
    If assigned(fDesktopName)and(not Application.Terminated) then NotDeskName:=W<>fDesktopName.Handle else NotDeskName:=true;
    If (W<>Handle)and(W<>Application.Handle)and(NotDeskName) Then
      If (IsWindowVisible(W)or(FindHandleInHiddenList(W)>=0)) Then Begin
         WndTitle:=GetWindowTitle(W);
         If WndTitle='Program Manager' Then WndTitle:='';
         If WndTitle<>'' Then WindowsList.Add(Pointer(W));
      End;
    W:=GetWindow(W,GW_HWNDNEXT);
  End;
End;

procedure TfMain.OnlyFromHere;
Var
  I: Integer;
Begin
  I:=0; While WindowsList.Count-1>I Do Begin
    If CountDesktopNo(Integer(WindowsList.Items[I]))=Desk Then Inc(I)
    Else WindowsList.Delete(I);
  End;
End;

procedure TfMain.MoveWindows(X,Y:Integer); 
Var
  I       : Word;
  WRct    : TRect;
  W       : THandle;
  MWStruct: HDWP;
  WP      : TWindowPlacement;
Begin
  GetVisibleWindows;

  If WindowsList.Count>0 Then Begin
    MWStruct:=BeginDeferWindowPos(WindowsList.Count-1);
    If Pointer(MWStruct)<>nil Then Begin
      For I:=0 To WindowsList.Count-1 Do Begin
        W := THandle(WindowsList[I]);
        GetWindowRect(W,WRct);
        GetWindowPlacement(W, @WP);
        If WP.ptMinPosition.X>-10000 Then Begin
          WP.ptMinPosition.X:=-10000;
          WP.ptMinPosition.Y:=-10000;
        End;
        If (abs(WRct.Left)<abs(WP.ptMinPosition.X))and
           (abs(WRct.Top)<abs(WP.ptMinPosition.Y))and
           not (CFG.StickyWindows.List.IsSticky(W)) Then Begin
          MWStruct:=DeferWindowPos(MWStruct, W, HWND_BOTTOM,
                      WRct.Left+X, WRct.Top+Y, WRct.Right-WRct.Left, WRct.Bottom-WRct.Top,
                      SWP_NOACTIVATE or SWP_NOZORDER);
          {$IFDEF SDEBUG}try Assert(False) except on E:EAssertionFailed do DebugWAddEString(E.Message, 'MoveWindows, %d "%s" %d:%d => %d:%d', [W, GetWindowTitle(W), WRct.Left, WRct.Top, WRct.Left+X, WRct.Top+Y]); end;{$ENDIF}
        End Else
          {$IFDEF SDEBUG}try Assert(False) except on E:EAssertionFailed do DebugWAddEString(E.Message, 'MoveWindows, skiping window %d "%s", propably hidden %d:%d', [W, GetWindowTitle(W), WRct.Left, WRct.Top]); end;{$ENDIF}
      End;
      EndDeferWindowPos(MWStruct);
    End;
  End;
End;

procedure TfMain.RealocateRemeberActive;
Var
  I: Integer;
begin
      if FRemeberActiveWnd.Count<>CFG.Window.Columns*CFG.Window.Rows then begin
        FRemeberActiveWnd.Clear;
        for I:=0 To CFG.Window.Columns*CFG.Window.Rows-1 do FRemeberActiveWnd.Add(nil);
      end;
end;

procedure TfMain.WDesk(aDesk:Word);
Var
  DX, DY : Integer;
  I      : Integer;
  hWnd   : THandle;
Begin
  hWnd := 0;

  If FDesk<>aDesk Then Begin
    {$IFDEF SDEBUG}try Assert(False) except on E:EAssertionFailed do DebugWAddEString(E.Message, 'WDesk, Switching from desktop %d to %d', [FDesk, aDesk]); end;{$ENDIF}
    // Check if isn't required range out of desktop
    If aDesk>(CFG.Window.Columns*CFG.Window.Rows) Then Exit;
    // Switch!
    DX := ((FDesk mod CFG.Window.Columns)-(aDesk mod CFG.Window.Columns))*Screen.Width;
    DY := ((FDesk div CFG.Window.Columns)-(aDesk div CFG.Window.Columns))*Screen.Height;

    { 07.08.1999 Code for handling "remeber active feature" by Jan Tomasek }
    if not Application.Terminated then begin
      RealocateRemeberActive;
      hWnd:=0;
      if (FDesk<FRemeberActiveWnd.Count)and(aDesk<FRemeberActiveWnd.Count) then begin
        hWnd                     := GetForegroundWindow;
        if (not CFG.DesktopNames.Enabled)or
           (CFG.DesktopNames.Enabled and (not fDesktopName.HideTimer.Enabled)) then
          FRemeberActiveWnd[FDesk] := Pointer(hWnd);
        {$IFDEF SDEBUG}try Assert(False) except on E:EAssertionFailed do DebugWAddEString(E.Message, 'WDesk, saving focus for window for %d = "%s", on desk %d', [hWnd, GetWindowTitle(hWnd), FDesk]); end;{$ENDIF}
        hWnd                     := THandle(FRemeberActiveWnd[aDesk]);
        {$IFDEF SDEBUG}try Assert(False) except on E:EAssertionFailed do DebugWAddEString(E.Message, 'WDesk, window for %d = "%s", will get focus', [hWnd, GetWindowTitle(hWnd), FDesk]); end;{$ENDIF}
      end;
    end; { end Jan }

    if CFG.DesktopNames.Enabled then begin
      fDesktopName.HideTimer.Enabled:= False;
      fDesktopName.Hide;
    end;

    { 16.07.99 Added by Tim Boughen }
    If CFG.DesktopSwitching.HideApps then ResetWindowStyles(aDesk);
    { End Tim }

    MoveWindows(DX,DY);

    FDesk:=aDesk;

    // Repaint
    Invalidate;
    // Show
    HideTimer.Enabled:=False; HideTimer.Enabled:=True; // restart
    UnDockWindow;
  End;

  { 07.08.1999 Code for handling "remeber active feature" by Jan Tomasek }
  if not Application.Terminated then begin
    if aDesk<FRemeberActiveWnd.Count then begin
      if (not IsWindow(hWnd))or(hWnd=Handle) then begin { window is not valid we will try to locate any other valid }
        for I:=WindowsList.Count-1 downto 0 do
          if (CountDesktopNo(THandle(WindowsList[I]))=FDesk)and(fDesktopName.Handle<>THandle(WindowsList[I])) then begin
            hWnd := THandle(WindowsList[I]);
            {$IFDEF SDEBUG}try Assert(False) except on E:EAssertionFailed do DebugWAddEString(E.Message, 'WDesk, window %d = "%s", will get focus on desk %d', [hWnd, GetWindowTitle(hWnd), aDesk]); end;{$ENDIF}
          end;
      end;

      if IsWindow(hWnd) then begin
        SetForegroundWindow(Handle);
        if not CFG.DesktopNames.Enabled then SetForegroundWindow(hWnd);
        {$IFDEF SDEBUG}try Assert(False) except on E:EAssertionFailed do DebugWAddEString(E.Message, 'WDesk, restoring focus for %d = "%s", on desk %d', [hWnd, GetWindowTitle(hWnd), aDesk]); end;{$ENDIF}
      end;
    end;
  end; { end Jan }

  if CFG.DesktopNames.Enabled and (not Application.Terminated) then begin
    fDesktopName.ForegroundWindow := hWnd;
    if CFG.DesktopNames.Names.Count>FDesk then fDesktopName.Caption := CFG.DesktopNames.Names[FDesk]
    else fDesktopName.Caption := Format('Desktop %d', [FDesk+1]);
    fDesktopName.HideTimer.Interval := CFG.DesktopNames.Delay;
    fDesktopName.Show;
  end;
End;

Procedure TfMain.MyMoveWindow(W:THandle;aDesk:Word); 
Var
  S      : ShortString;
  DX, DY : Integer;
  _X, _Y : Integer;
  CRct   : TWindowPlacement;
  WRct   : TRect;
Begin
  S:=GetWindowTitle(W);
  GetWindowRect(W,WRct);
  GetWindowPlacement(W, @CRct);

  Inc(WRct.Left, 5); Inc(WRct.Top, 5); // For maximized windows
  DX    := (aDesk mod CFG.Window.Columns)-(FDesk mod CFG.Window.Columns);
  DY    := (aDesk div CFG.Window.Columns)-(FDesk div CFG.Window.Columns);
  While(WRct.Left<0)and(Abs(WRct.Left)>(Screen.Width*0.6))Do Inc(WRct.Left, Screen.Width);
  _X    := WRct.Left mod Screen.Width -5;
  While(WRct.Top<0)and(Abs(WRct.Top)>(Screen.Height*0.6))Do Inc(WRct.Top, Screen.Height);
  _Y    := WRct.Top mod Screen.Height -5;

  GetWindowRect(W,WRct);
  MoveWindow(W, {WHND}
             _X+DX*Screen.Width,
             _Y+DY*Screen.Height, {Position}
             WRct.Right-WRct.Left, WRct.Bottom-WRct.Top, {Width}
             True); {Repaint}
  {$IFDEF SDEBUG}try Assert(False) except on E:EAssertionFailed do DebugWAddEString(E.Message, 'MyMoveWindow, , %d "%s" => %d:%d', [W, GetWindowTitle(W), _X+DX*Screen.Width, _Y+DY*Screen.Height]); end;{$ENDIF}
//  GetWindowRect(W,WRct);
  FormPaint(self);
End;

Procedure TfMain.MoveHereClick(Sender: TObject);
Var
  I   : Integer;
  aMI,
  oMI : TMenuItem;
Begin
  aMI:=(Sender as TMenuItem);

  If aMI.Tag<>-1 Then Begin
    if not FPopMenuFromTrayIcon then
      MyMoveWindow(THandle(aMI.Tag), Trunc(popX/CFG.Window.xStep)+Trunc(popY/CFG.Window.yStep)*CFG.Window.Columns)
    else
      MyMoveWindow(THandle(aMI.Tag), FDesk);
    SetForegroundWindow(THandle(aMI.Tag));
  End Else Begin
    oMI:=(aMI.Owner as TMenuItem);
    For I:=0 To oMI.Count-1 Do
      If oMI.Items[I].Tag>0 Then MyMoveWindow(THandle(oMI.Items[I].Tag), Trunc(popX/CFG.Window.xStep)+Trunc(popY/CFG.Window.yStep)*CFG.Window.Columns);
  End;
  If CFG.DesktopSwitching.HideApps then ResetWindowStyles(FDesk);

  FPopMenuFromTrayIcon := False;
End;

Procedure TfMain.SwitchToClick(Sender: TObject);
Begin
  Desk:=CountDesktopNo((Sender as TComponent).Tag);
  SetForegroundWindow((Sender as TComponent).Tag);
End;

procedure TfMain.FormMouseUp(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
Var
  aDesk : Integer;
begin
  IgnoreSetFocus:=True;
  AllowMove.Enabled:=False;

  If Button=mbLeft Then Begin
    If (not MovingWindow)or(MovingWindow and(MWStartMPos.X=X)and(MWStartMPos.Y=Y)) Then Begin
      if MWindow<>-1 then begin
        RealocateRemeberActive;
        if (not CFG.DesktopNames.Enabled)or
           (CFG.DesktopNames.Enabled and (not fDesktopName.HideTimer.Enabled)) then
          FRemeberActiveWnd[FDesk] := Pointer(MWindow);
      end;

      aDesk := Trunc(X/CFG.Window.xStep)+Trunc(Y/CFG.Window.yStep)*CFG.Window.Columns;
      Desk  := aDesk;

{      if aDesk=Desk then
        if MWindow<>-1 then SetForegroundWindow(MWindow);}

{      If MWindow=-1 Then Begin
        GetVisibleWindows;
        OnlyFromHere;
        If WindowsList.Count>0 Then SetForegroundWindow(Integer(WindowsList.Items[0]))
        Else SetForegroundWindow(Handle);
      End;}
      AllowDeskSwitch:=False;
    End;
    MovingWindow:=False;
    Invalidate;
  End Else If Button=mbRight Then Begin

  End;

  HideTimer.Enabled:=False; HideTimer.Enabled:=True; // restart
  PostMessage(Handle, UM_ENABFOCUS, 0, 0);
end;

procedure TfMain.UpdateTimerTimer(Sender: TObject);
Var
  W          : Word;
begin
  W:=GetAsyncKeyState(VK_SCROLL);
  If Word(W and $E000)=$E000 Then Beep;

  FormPaint(self);
end;

Function IsMaximized(W:HWND):Boolean;
Var
  WP : TWindowPlacement;
Begin
  FillChar(WP, SizeOf(WP), 0); WP.length:=SizeOf(WP);
  GetWindowPlacement(W, @WP);
  Result:=WP.flags and WPF_RESTORETOMAXIMIZED = WPF_RESTORETOMAXIMIZED;
End;

procedure TfMain.FormMouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
Var
  I       : integer;
  WRct    : TRect;
  W       : THandle;
  MenuI   : TMenuItem;
  Root    : PWin2TreeElement;
begin
  // MWindow is window which we are moving
  MWindow:=-1;
  AllowMove.Tag:=0;

  // left button - user want:
  //     * switch desktop
  //     * move window
  If Button=mbLeft Then Begin
    // Get list of all visible windows
    GetVisibleWindows;
    // Flag for OnMouseUp event: True ... move; False ... switch desktop
    MovingWindow:=False;
    // Windows are in some specific order (Z-order ?), we will move first which
    // is under muse cursor
    For I:=0 To WindowsList.Count-1 Do Begin
      W := THandle(WindowsList[I]);
      GetWindowRect(W,WRct);
      // Transform from real coordinates to sDesk coordinates
      Inc(WRct.Left,  (FDesk mod CFG.Window.Columns)*Screen.Width);
      Inc(WRct.Right, (FDesk mod CFG.Window.Columns)*Screen.Width);

      Inc(WRct.Top,   (FDesk div CFG.Window.Columns)*Screen.Height);
      Inc(WRct.Bottom,(FDesk div CFG.Window.Columns)*Screen.Height);

      // Check if transformed mouse cursor coordinates are in transformed
      // window coordinates.
      If IsInRect(WRct, Round(X*Screen.Width/CFG.Window.xStep), Round(Y*Screen.Height/CFG.Window.yStep))and
         not (IsMaximized(W) and not CFG.Window.AllowMoveMaximized) Then Begin
        GetWindowRect(W,WRct);
        MovingWindow      :=True;
        MWindow           :=W; // store handle of window which we want to move
                               // for OnMouseMove and OnMouseUp event
        AllowMove.Interval:= CFG.Window.MoveTimeout;
        AllowMove.Enabled := True; // moving is allowed after timeout
        MWStartMPos       := Point(X, Y);
        MWStartWPos       := Point(WRct.Left, WRct.Top);
        Break;
      End;
    End;
    {$IFDEF SDEBUG}try Assert(False) except on E:EAssertionFailed do DebugWAddEString(E.Message, 'LeftMouseDown, MWindow=%d "%s"', [MWindow, GetWindowTitle(MWindow)]); end;{$ENDIF}
  End Else
   // right mouse button user want to see menu
   If Button=mbRight Then Begin
     HideTimer.Enabled:=True;
     // Clear menu
     While pmContextMenu.Items[0].Count>0 Do Begin // Move here
       MenuI:=pmContextMenu.Items[0].Items[0];
       pmContextMenu.Items[0].Remove(MenuI);
     End;
     While pmContextMenu.Items[1].Count>0 Do Begin // Switch to
       MenuI:=pmContextMenu.Items[1].Items[0];
       pmContextMenu.Items[1].Remove(MenuI);
     End;
     While pmContextMenu.Items[2].Count>0 Do Begin // Sticky windows
       MenuI:=pmContextMenu.Items[2].Items[0];
       pmContextMenu.Items[2].Remove(MenuI);
     End;
     // Build menu
     Root:=LoadTree;
     BuildMenu(Root, miMoveHere, MoveHereClick, False, nil);
     BuildMenu(Root, miSwitchTo, SwitchToClick, True, nil);
     BuildMenu(Root, miStickyWindows, StickyWindowsClick, True, CFG.StickyWindows.List);
     DestroyTree(Root);
     // miMoveHere.Enabled:=(X>0)and(Y>0);
     // Dispay menu
     pmContextMenu.Popup(Left+X,Top+Y);
     popX:=X;
     popY:=Y;
  End;
end;

procedure TfMain.FormMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer); 
Var
  WRct  : TRect;
  MoveTo: TPoint;
begin
  If MovingWindow Then Begin
    If((Y>0)and(X>0)and(Y<ClientHeight)and(X<ClientWidth))and(AllowMove.Tag=1) Then Begin
      { 16.07.99 Added by Tim Boughen }
      If CFG.DesktopSwitching.HideApps then ResetWindowStyles(FDesk);
      { End Tim }
      GetWindowRect(MWindow,WRct);
      MoveTo := Point(MWStartWPos.X-Round((MWStartMPos.X-X)*Screen.Width/CFG.Window.xStep),
                      MWStartWPos.Y-Round((MWStartMPos.Y-Y)*Screen.Height/CFG.Window.yStep));
      MoveWindow(MWindow, MoveTo.X, MoveTo.Y,
                 WRct.Right-WRct.Left, WRct.Bottom-WRct.Top, True);
      FormPaint(self);
      HideTimer.Enabled:=False;
      {$IFDEF SDEBUG}try Assert(False) except on E:EAssertionFailed do DebugWAddEString(E.Message, 'MouseMove, MWindow=%d "%s" => %d:%d', [MWindow, GetWindowTitle(MWindow), MoveTo.X, MoveTo.Y]); end;{$ENDIF}
      Exit;
    End
  End Else Begin
    UnDockWindow;
    HideTimer.Enabled:=False; HideTimer.Enabled:=True;
  End;
end;

procedure TfMain.HideTimerTimer(Sender: TObject);
Var
  P  : TPoint;
begin
  GetCursorPos(P);
  If IsInRect(Rect(Left, Top, Left+Width, Top+Height), P.X, P.Y) Then Exit;

  HideTimer.Enabled:=False;
  If CFG.Window.Docking Then While DockWindow Do Sleep(10);
end;

Procedure TfMain.WMNCMOUSEMOVE(Var Msg:TMsg);
Begin
  HideTimer.Enabled:=False; HideTimer.Enabled:=True; // restart

  UnDockWindow;
End;

procedure TfMain.miOptionsClick(Sender: TObject);
var
  AppsHidden : boolean;
begin
  AppsHidden:=CFG.DesktopSwitching.HideApps;
  OptionsProc:=True;
  try
    URegKeys;
    fOptions.Data:=CFG;
    If fOptions.ShowModal=mrOk Then Begin
      CFG:=fOptions.Data;
      TrayIcon.Active := CFG.General.PutIconToTray;
      If CFG.Window.ShowTitleBar Then BorderStyle := bsSizeToolWin
      Else BorderStyle := bsNone;
      ClientWidth := CFG.Window.Columns*CFG.Window.xStep;
      ClientHeight:= CFG.Window.Rows*CFG.Window.yStep;
      if not CFG.DesktopSwitching.HideApps and AppsHidden then RestoreHiddenApps;
    End;
    SetParams;
    WriteCFG(CFG);
    HideTimer.Enabled:=True;
    UnDockWindow;
  finally
    OptionsProc:=False;
  end;
end;

procedure TfMain.FormShow(Sender: TObject);
begin
//  ClientWidth := CFG.Window.Columns*CFG.Window.xStep;
//  ClientHeight:= CFG.Window.Rows*CFG.Window.yStep;
  UnDockWindow;
end;

// -----------------------------------------------------------------------------
// This function return number of desktop where is given window
function  TfMain.CountDesktopNo(W:THandle):Integer; register; 
Var
  WRct : TRect;
  X, Y : Integer;
Begin
  // Get window's coordinates
  GetWindowRect(W, WRct);

  // Get center point of window
  X    := (WRct.Left+WRct.Right)div 2;
  Y    := (WRct.Top+WRct.Bottom)div 2;

  X    := (FDesk mod CFG.Window.Columns)*Screen.Width +X;
  Y    := (FDesk div CFG.Window.Columns)*Screen.Height+Y;

  Result:=(X div Screen.Width)+(Y div Screen.Height)*CFG.Window.Columns;
End;

// -----------------------------------------------------------------------------
// This is "filter" function I need to ignore sDesk and Program manager (name
// of explorer window).
{function TfMain.IsValidWindow(W:THandle):boolean;
Var
  S    : ShortString;
Begin
  S:=GetWindowTitle(W);
  Result:=not ((S=Caption)or(S='Program Manager'));
End;}

// -----------------------------------------------------------------------------
// Return true if screen saver is active
(*function  TfMain.IsScreenSaverActive:Boolean;
Begin
  // WindowsScreenSaverClass:Screen Saver [0:0 800:600], ICON=1836198, 5
  Result:=(FindWindow('WindowsScreenSaverClass', 'Screen Saver')>0) or // only this is correct
          (FindWindow(nil, 'Screen Saver')>0) or
          (FindWindow('WindowsScreenSaverClass', nil)>0) or
          (FindWindow('WindowsScreenSaverClass', 'WindowsScreenSaverClass')>0);
End;*)

Procedure TfMain.WMHOTKEY(Var Msg:TMsg);
Var
  P     : TPoint;
Begin
  Case Msg.message of
    hkDeskPlus  : Begin
      If ((Desk mod CFG.Window.Columns)+1)<CFG.Window.Columns Then
        Desk:=Desk+1
      Else
        If CFG.General.CycleDesktops Then
          { 19/07/99 Added by Tim Boughen }
          If CFG.General.CycleSingleRow Then
             Desk:=Desk-(Desk mod CFG.Window.Columns)
          Else
             If (Desk+1) = (CFG.Window.Columns*CFG.Window.Rows) Then Desk:=0 Else Desk:=Desk+1
          { End Tim }
        Else
          MessageBeep(1);
    End;
    hkDeskMinus : Begin
      If Desk mod CFG.Window.Columns>0 Then
        Desk:=Desk-1
      Else
        If CFG.General.CycleDesktops Then
          { 19/07/99 Added by Tim Boughen }
          IF CFG.General.CycleSingleRow Then
             Desk:=(Desk div CFG.Window.Rows +1)*CFG.Window.Columns-1
          Else
             If Desk=0 Then Desk:=(CFG.Window.Columns*CFG.Window.Rows) -1 Else Desk:=Desk-1
          { End Tim }      
        Else
          MessageBeep(1);
    End;
    hkDeskPlusRow  : Begin
      { 19/07/99 Modified by Tim Boughen
        MUST TEST <= }
      If (Desk+CFG.Window.Columns+1)<=(CFG.Window.Columns*CFG.Window.Rows) Then
        Desk:=Desk+CFG.Window.Columns
      Else
        If CFG.General.CycleDesktops Then
          Desk:=(Desk+CFG.Window.Columns)mod(CFG.Window.Columns*CFG.Window.Rows)
        Else
          MessageBeep(1);
    End;
    hkDeskMinusRow : Begin
      If Desk>=CFG.Window.Columns Then
        Desk:=Desk-CFG.Window.Columns
      Else
        If CFG.General.CycleDesktops Then Begin
          Desk:=(Desk mod CFG.Window.Columns)+CFG.Window.Columns*(CFG.Window.Rows-1);
        End Else
          MessageBeep(1);
    End;
    hkContextMenu: Begin
      FPopMenuFromTrayIcon := True;
      GetCursorPos(P);
      UnDockWindow;
      SetForegroundWindow(Handle);
      FormMouseDown(Self, mbRight, [], P.X-Left-5, P.Y-Top-5);
    End;
  End;
  If Msg.Message in [hkDesk..hkDesk+CFG.Window.Columns*CFG.Window.Rows-1] Then
    Desk:= Msg.Message-hkDesk;
End;

procedure TfMain.WndProc(var Message: TMessage); 
Begin
  Case message.Msg of
    WM_SHOWWINDOW,
    WM_ACTIVATE             : If CFG.General.HideFromTaskBar Then Begin
                                 ShowWindow(Application.Handle, SW_HIDE);
                                 If not CFG.Window.ShowTitleBar Then Begin
                                    SetWindowLong(Application.Handle, GWL_EXSTYLE, WS_EX_TOOLWINDOW);
                                    end;
                                 end;
    UM_ENABFOCUS            : IgnoreSetFocus:=False;
    136                     : IsScreenSaverActive := False;
    WM_TIMER                : ;
    WM_GETTEXT              : ;
  end;

  Inherited WndProc(Message);
End;

procedure TfMain.miAboutClick(Sender: TObject);
begin
  fAboutBox.ShowModal;
end;

procedure TfMain.TrayIconClick(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
Var
  P : TPoint;
begin
  { 13.07.1999 added by Tim Boughen, Jan Tomasek}
  If Button=mbRight Then Begin
    GetCursorPos(P);                 // Get coordinates of mouse's cursor
    SetForegroundWindow(Handle);     // If this isn't set, context menu
                                     // didn't disapear after click out of it.
    FPopMenuFromTrayIcon := True;
    FormMouseDown(Sender, mbRight, [], P.X-Left-5, P.Y-Top-5);
  End Else Begin
  { end Tim }
    HideTimer.Enabled:=False; HideTimer.Enabled:=True; // restart
    UnDockWindow;
    Show;
  End;
end;

procedure TfMain.KeyAckTimer(Sender: TObject);
begin
{  KeyAck.Enabled:=KeyAck.Tag>0;
  If GetAsyncKeyState(fOptions.AutoKeyAck)<0 Then Begin
    Desk:=CountDesktopNo(GetForegroundWindow);
    KeyAck.Enabled:=False;
    KeyAck.Tag:=0;
  End;
  If KeyAck.Enabled Then KeyAck.Tag:=KeyAck.Tag-1;}
end;

procedure TfMain.UnDockWindow;
Begin
  If CFG.Window.Docking Then Begin
    Case CFG.Window.DockingPos of
      sDeskTOP    : Begin
        Top:=0;
        If (Left+Width+5)>Screen.Width Then Left:=Screen.Width-Width;
        If (Left+Width+5)<0 Then Left:=0;
      End;
      sDeskBOTTOM : Begin
        Top:=Screen.Height-Height;
        If (Left+Width+5)>Screen.Width Then Left:=Screen.Width-Width;
        If (Left+Width+5)<0 Then Left:=0;
      End;
      sDeskLEFT   : Begin
        Left:=0;
        If Top<0 Then Top:=0;
        If Top>Screen.Height Then Top:=Screen.Height-Height;
      End;
      sDeskRIGHT  : Begin
        Left:=Screen.Width-Width;
        If Top<0 Then Top:=0;
        If Top>Screen.Height Then Top:=Screen.Height-Height;
      End;
    End;
  End Else Begin
    // Ckeck if a part of window is visible
  End;
End;

function TfMain.DockWindow:Boolean;
Var
  VisiblePart : Integer;
Begin
  If CFG.Window.Docking Then Begin
    // Continue docking
    Result := True;

    If not CFG.Window.NotHide Then
      VisiblePart := CFG.Window.VisiblePart
    Else Begin
      If CFG.Window.DockingPos in [sDeskTOP, sDeskBOTTOM] Then
        VisiblePart := Height
      Else
        VisiblePart := Width;
    End;

    Case CFG.Window.DockingPos of
      sDeskTOP    : Begin
        Top:=Top-DockingStep;
        If (Top<=VisiblePart-Height)or not CFG.Window.SmoothHiding Then Begin
          Top:=VisiblePart-Height;
          Result:=False;
        End;
      End;
      sDeskBOTTOM : Begin
        Top:=Top+DockingStep;
        If (Top>Screen.Height-VisiblePart)or not CFG.Window.SmoothHiding Then Begin
          Top:=Screen.Height-VisiblePart;
          Result:=False;
        End;
      End;
      sDeskLEFT   : Begin
        Left:=Left-DockingStep;
        If (Left+Width<=VisiblePart)or not CFG.Window.SmoothHiding Then Begin
          Left:=VisiblePart-Width;
          Result:=False;
        End;
      End;
      sDeskRIGHT  : Begin
        Left:=Left+DockingStep;
        If (Left>Screen.Width-VisiblePart)or not CFG.Window.SmoothHiding Then Begin
          Left:=Screen.Width-VisiblePart;
          Result:=False;
        End;
      End;
    End;
  End Else Begin
    // Ckeck if a part of window is visible
    Result := False;
  End;
End;

procedure TfMain.ScreenSaverTimerTimer(Sender: TObject);
Var
  P          : TPoint;
  B          : bool;
  ExecSC     : Boolean;
  SamePoint  : Boolean;
begin
  // Get position of mouse cursor
  GetCursorPos(P);
  // Check if mouse is in selected rectangle
  Case CFG.ScreenSaver.ActivationPos of
    sDeskLEFTTOP     : ExecSC := IsInRect(Rect(0, 0, 0+CFG.ScreenSaver.Sensitivity, 0+CFG.ScreenSaver.Sensitivity), P.X, P.Y);
    sDeskRIGHTTOP    : ExecSC := IsInRect(Rect(Screen.Width-CFG.ScreenSaver.Sensitivity, 0, Screen.Width, 0+CFG.ScreenSaver.Sensitivity), P.X, P.Y);
    sDeskLEFTBOTTOM  : ExecSC := IsInRect(Rect(0, Screen.Height-CFG.ScreenSaver.Sensitivity, 0+CFG.ScreenSaver.Sensitivity, Screen.Height), P.X, P.Y);
    sDeskRIGHTBOTTOM : ExecSC := IsInRect(Rect(Screen.Width-CFG.ScreenSaver.Sensitivity, Screen.Height-CFG.ScreenSaver.Sensitivity, Screen.Width, Screen.Height), P.X, P.Y);
    Else               ExecSC := False
  End;
  // Is allowed using screen saver
  SystemParametersInfo(SPI_GETSCREENSAVEACTIVE, 0, @B, 0);
  // Check if mouse was moved
  If not ExecSC Then ScreenSaverLastPos := P;
  SamePoint:=(P.x=ScreenSaverLastPos.x)and(P.y=ScreenSaverLastPos.y);
  // If all conditions are pased start it!
  If (ExecSC and B and not IsScreenSaverActive)and
     (CFG.ScreenSaver.AllowRerun or not SamePoint) Then Begin
    Inc(ScreenSaverTickCount);
    If ScreenSaverTickCount>=10 Then Begin
      ScreenSaverLastPos := P;
      If CFG.ScreenSaver.RunFile Then Begin
        IsScreenSaverActive := True;
        WinExecAndWait32(CFG.ScreenSaver.RunFileName+' '+CFG.ScreenSaver.RunFileArgs, 1);
        IsScreenSaverActive := False;
      End Else Begin
        PostMessage(GetDesktopWindow, WM_SYSCOMMAND, SC_SCREENSAVE, 0);
        IsScreenSaverActive := True;
      End;
    End;
  End Else Begin
    ScreenSaverTickCount:=0;
  End;
end;

procedure TfMain.AllowMoveTimer(Sender: TObject);
begin
  {$IFDEF SDEBUG}try Assert(False) except on E:EAssertionFailed do DebugWAddEString(E.Message, 'AllowMoveTimer, timeout => move allowed MWindow=%d "%s"', [MWindow, GetWindowTitle(MWindow)]); end;{$ENDIF}
  AllowMove.Enabled := False;
  AllowMove.Tag     := 1;
end;

procedure TfMain.AutoSwitchTimer(Sender: TObject);
Var
  KeyState:Byte;
begin
  If GetForegroundWindow=LastWindowHandle Then AutoSwitch.Tag:=AutoSwitch.Tag+1
  Else Begin
    AutoSwitch.Tag:=0;
    If GetForegroundWindow<>Handle Then AllowDeskSwitch:=True;
    LastWindowHandle:=GetForegroundWindow;
  End;

  Case CFG.DesktopSwitching.SwitchMode of
    sDeskSwitchNONE: ;
    sDeskSwitchAUTO: If (AutoSwitch.Tag>5)and(LastWindowHandle<>Handle)and
         (FDesk<>CountDesktopNo(LastWindowHandle))and AllowDeskSwitch Then Begin
      SetForegroundWindow(Handle);
      Desk:=CountDesktopNo(LastWindowHandle);
    End;
    sDeskSwitchMANUAL: If (AutoSwitch.Tag<15)and(LastWindowHandle<>Handle)and
         (FDesk<>CountDesktopNo(LastWindowHandle)) Then Begin
      KeyState:=0;
      Case CFG.DesktopSwitching.SwitchKey of
        sDeskSwitchALT   : KeyState:=GetAsyncKeyState(VK_MENU);
        sDeskSwitchSHIFT : KeyState:=GetAsyncKeyState(VK_SHIFT);
        sDeskSwitchCTRL  : KeyState:=GetAsyncKeyState(VK_CONTROL);
      End;
      If KeyState>0 Then Begin
//        This Sets foreground to some app when desktop switched.
//        Otherwise, Foreground window is still the last active one.
//        SetForegroundWindow(Handle);
        Desk:=CountDesktopNo(LastWindowHandle);
      End;
    End;
  End;
end;

Procedure TfMain.MoveAllToFirst;
Var
  I: Integer;
Begin
  GetVisibleWindows;
  For I:=0 To WindowsList.Count-1 Do MyMoveWindow(THandle(WindowsList[I]), 0);
End;

procedure TfMain.miExitClick(Sender: TObject);
begin
  Close;
End;

procedure TfMain.StickyWindowsClick(Sender: TObject);
Var
  mi : TMenuItem;
begin
  mi := Sender as TMenuItem;
  If mi.Checked Then Begin
    CFG.StickyWindows.List.Remove(mi.Tag)
  End Else Begin
    If GetAsyncKeyState(VK_CONTROL)<>0 Then Begin
      fStickyProp.Mode           := spmContextMenu;
      fStickyProp.WindClassName  := GetWindowClass(mi.Tag);
      fStickyProp.PartOfTitle    := GetWindowTitle(mi.Tag);
      If fStickyProp.ShowModal=mrOk Then Begin
        CFG.StickyWindows.List.Add(
          TStickyElement.Create(mi.Tag, fStickyProp.ClassName,
            fStickyProp.PartOfTitle, fStickyProp.StickyMode));
      End;
    End Else Begin
      CFG.StickyWindows.List.Add(TStickyElement.Create(mi.Tag, '', '', semHandle));
    End;
  End;
end;

Function THiddenListItem.RTitle:String;
Begin
  Result:=GetWindowTitle(FWndHandle);
End;

procedure TfMain.FormCloseQuery(Sender: TObject; var CanClose: Boolean);
begin
  Desk:=0; // Move to first desktop

  If CFG.General.MoveToFirst Then MoveAllToFirst;

  { 13.07.1999 added by Tim Boughen }
  RestoreHiddenApps;
  { end Tim }
end;

procedure TfMain.MouseSwitchTimer(Sender: TObject);
Var
  Point: TPoint;
  MovedPoint: TPoint;
  LastDesk: Integer;
  Msg  : TMsg;
  Sens : Integer;

begin
  if (not CFG.DesktopSwitching.MouseSwEnabled)and(MouseSwitch.Enabled) then begin
    MouseSwitch.Enabled := False;
    Exit;
  end;
  try
    MouseSwitch.Enabled := False;


    GetCursorPos(Point);
    if (Point.X=FLastCusorPos.X)and(Point.Y=FLastCusorPos.Y) then begin
      Inc(FMouseSwitchCounter);
    end else begin
      FLastCusorPos := Point;
      FMouseSwitchCounter := 0;
    end;

    if FMouseSwitchCounter>=3 then begin
      Sens                := CFG.DesktopSwitching.MouseSwSens+1;
      FMouseSwitchCounter := 0;
      MovedPoint          := Point;
      LastDesk            := Desk;

      if (Point.x<Sens)and(Point.y>Sens)and(Screen.Height-1-Sens>Point.y) then begin
        { Left }
        Msg.message := hkDeskMinus;
        WMHOTKEY(Msg);
        MovedPoint.X := Screen.Width-Sens;
      end else if (Point.x>Screen.Width-Sens)and(Point.y>Sens)and(Screen.Height-1-Sens>Point.y) then begin
        { Right }
        Msg.message := hkDeskPlus;
        WMHOTKEY(Msg);
        MovedPoint.X := Sens;
      end else if (Point.y>Screen.Height-Sens)and(Point.x>Sens)and(Screen.Width-1-Sens>Point.x) then begin
        { Bottom }
        Msg.message := hkDeskPlusRow;
        WMHOTKEY(Msg);
        MovedPoint.Y := Screen.Height-Sens;
      end else if (Point.y<Sens)and(Point.x>Sens)and(Screen.Width-1-Sens>Point.x) then begin
        { Top }
        Msg.message := hkDeskMinusRow;
        WMHOTKEY(Msg);
        MovedPoint.Y := Sens;
      end;

      if LastDesk<>Desk then
        if CFG.DesktopSwitching.MouseSwMoveCur then
          SetCursorPos(MovedPoint.X, MovedPoint.Y);
    end;
  finally
    MouseSwitch.Enabled := True;
  end;
end;

end.

